# 📚 Panduan Lengkap Hosting Website Rental Mobil

## 🎯 Overview
Website rental mobil Anda sudah siap untuk di-deploy ke production. Berikut adalah panduan lengkap untuk berbagai platform hosting.

---

## 🔧 1. Persiapan Sebelum Deploy

### A. Build Production
```bash
# Build untuk production
npm run build

# Test build locally
npm run start
```

### B. Environment Variables
Buat file `.env.production`:
```env
# Database
DATABASE_URL="file:./prod.db"

# Next.js
NEXTAUTH_URL="https://domainanda.com"
NEXTAUTH_SECRET="your-secret-key-here"

# API Configuration
NODE_ENV="production"
```

### C. File yang Diperlukan
- `next.config.js` - Konfigurasi Next.js
- `package.json` - Dependencies dan scripts
- `public/` - Static assets (logo, favicon, images)
- `.next/` - Build hasil (generate otomatis)
- `prisma/` - Database schema
- `db/` - Database file (SQLite)

---

## 🌐 2. Panduan Hosting per Platform

### A. cPanel (Shared Hosting)

#### 1. Persiapan di cPanel
```bash
# 1. Upload file via File Manager atau FTP
# 2. Extract ke folder public_html atau subdomain
# 3. Pastikan file permissions:
#    - Folder: 755
#    - File: 644
```

#### 2. Setup Node.js di cPanel
```bash
# 1. Buka "Setup Node.js App" di cPanel
# 2. Pilih versi Node.js 18+
# 3. Upload project files
# 4. Set application root ke project folder
# 5. Set startup file: `package.json`
# 6. Set run command: `npm start`
```

#### 3. Konfigurasi Database SQLite
```bash
# 1. Buka "phpMyAdmin" atau "SQLite Manager"
# 2. Upload file database (.db) ke folder yang aman
# 3. Update DATABASE_URL di environment
```

#### 4. Domain & SSL
```bash
# 1. Setup domain/subdomain di cPanel
# 2. Enable SSL Certificate (Let's Encrypt)
# 3. Setup .htaccess untuk routing
```

### B. Vercel (Recommended)

#### 1. Deploy ke Vercel
```bash
# Install Vercel CLI
npm i -g vercel

# Login ke Vercel
vercel login

# Deploy
vercel --prod
```

#### 2. Konfigurasi Vercel
```json
// vercel.json
{
  "version": 2,
  "builds": [
    {
      "src": "package.json",
      "use": "@vercel/next"
    }
  ],
  "routes": [
    {
      "src": "/(.*)",
      "dest": "/$1"
    }
  ],
  "env": {
    "DATABASE_URL": "@database_url",
    "NEXTAUTH_SECRET": "@nextauth_secret"
  }
}
```

#### 3. Environment Variables di Vercel Dashboard
- Database URL
- NextAuth Secret
- Node Environment: production

### C. Netlify

#### 1. Deploy ke Netlify
```bash
# Install Netlify CLI
npm i -g netlify-cli

# Build
npm run build

# Deploy
netlify deploy --prod --dir=.next
```

#### 2. Konfigurasi Netlify
```toml
# netlify.toml
[build]
  publish = ".next"
  command = "npm run build"

[build.environment]
  NODE_VERSION = "18"

[[redirects]]
  from = "/*"
  to = "/index.html"
  status = 200
```

### D. DigitalOcean (VPS)

#### 1. Setup VPS
```bash
# 1. Create Droplet dengan Ubuntu 22.04
# 2. Setup SSH access
# 3. Install Node.js 18+
curl -fsSL https://deb.nodesource.com/setup_18.x | sudo -E bash -
sudo apt-get install -y nodejs
```

#### 2. Setup PM2 (Process Manager)
```bash
# Install PM2
npm install -g pm2

# Start aplikasi dengan PM2
pm2 start npm --name "rental-mobil" -- start

# Save PM2 configuration
pm2 save
pm2 startup
```

#### 3. Setup Nginx Reverse Proxy
```nginx
# /etc/nginx/sites-available/rental-mobil
server {
    listen 80;
    server_name domainanda.com;
    
    location / {
        proxy_pass http://localhost:3000;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_cache_bypass $http_upgrade;
    }
}
```

### E. AWS (Amazon Web Services)

#### 1. Deploy ke AWS Amplify
```bash
# Install Amplify CLI
npm install -g @aws-amplify/cli

# Initialize Amplify
amplify init

# Configure hosting
amplify add hosting

# Deploy
amplify publish
```

#### 2. AWS EC2
```bash
# 1. Launch EC2 instance
# 2. Setup security groups (port 3000)
# 3. Connect via SSH
# 4. Deploy application
git clone https://github.com/username/rental-mobil.git
cd rental-mobil
npm install
npm run build
pm2 start npm -- start
```

---

## 🔧 3. Konfigurasi Production

### A. Database Production
```sql
-- Generate Prisma Client untuk production
npx prisma generate

-- Push schema ke production database
npx prisma db push

-- Seed initial data (opsional)
npx prisma db seed
```

### B. Performance Optimization
```javascript
// next.config.js
/** @type {import('next').NextConfig} */
const nextConfig = {
  output: 'standalone',
  experimental: {
    optimizeCss: true,
    optimizePackageImports: ['lucide-react', '@radix-ui/react-icons']
  },
  images: {
    domains: ['domainanda.com'],
    formats: ['image/webp', 'image/avif']
  },
  compress: true
}

module.exports = nextConfig
```

### C. Security Configuration
```javascript
// Middleware untuk security
import { NextResponse } from 'next/server'

export function middleware(request) {
  // Security headers
  const response = NextResponse.next()
  
  response.headers.set('X-Frame-Options', 'DENY')
  response.headers.set('X-Content-Type-Options', 'nosniff')
  response.headers.set('Referrer-Policy', 'strict-origin-when-cross-origin')
  
  return response
}
```

---

## 📁 4. File Structure untuk Upload

```
rental-mobil/
├── public/                 # Static assets
│   ├── logo.svg
│   ├── favicon.ico
│   └── images/
├── src/                   # Source code
│   ├── app/
│   ├── components/
│   └── lib/
├── prisma/                # Database
│   └── schema.prisma
├── db/                    # SQLite database
├── package.json           # Dependencies
├── next.config.js        # Next.js config
├── tailwind.config.js     # Tailwind config
├── .env.production       # Environment variables
└── README.md             # Documentation
```

---

## 🚀 5. Script Deployment Otomatis

### A. Build Script
```json
// package.json
{
  "scripts": {
    "dev": "next dev",
    "build": "next build",
    "start": "next start",
    "lint": "next lint",
    "build:prod": "NODE_ENV=production next build",
    "deploy:vercel": "vercel --prod",
    "deploy:netlify": "npm run build && netlify deploy --prod",
    "deploy:cpanel": "npm run build && rsync -avz .next/ user@server:/path/to/app/"
  }
}
```

### B. Deploy Script (Shell)
```bash
#!/bin/bash
# deploy.sh

echo "🚀 Starting deployment..."

# Build application
echo "📦 Building application..."
npm run build

# Upload to server
echo "📤 Uploading files..."
rsync -avz --delete .next/ user@server:/var/www/rental-mobil/

# Restart application
echo "🔄 Restarting application..."
ssh user@server "pm2 restart rental-mobil"

echo "✅ Deployment completed!"
```

---

## 🔍 6. Testing & Monitoring

### A. Production Testing
```bash
# 1. Test semua endpoints
curl -X GET https://domainanda.com/api/cars
curl -X POST https://domainanda.com/api/bookings

# 2. Test database connection
npx prisma db pull

# 3. Test performance
lighthouse https://domainanda.com
```

### B. Error Monitoring
```javascript
// lib/monitoring.js
export function logError(error, context = {}) {
  console.error('Production Error:', {
    error: error.message,
    stack: error.stack,
    context,
    timestamp: new Date().toISOString()
  })
  
  // Send ke monitoring service
  // sendToSentry(error)
}
```

---

## 📊 7. Performance & SEO

### A. Meta Tags
```html
<!-- public/index.html atau di _document.tsx -->
<meta name="description" content="Rental mobil terpercaya dengan armada lengkap dan harga terjangkau">
<meta name="keywords" content="rental mobil, sewa mobil, car rental, mobil murah">
<meta property="og:title" content="Rental Mobil Professional">
<meta property="og:description" content="Layanan rental mobil terbaik dengan armada lengkap">
<meta property="og:image" content="https://domainanda.com/logo.svg">
```

### B. Sitemap
```javascript
// app/sitemap.xml/route.ts
export async function GET() {
  const cars = await db.car.findMany()
  
  return new Response(`<?xml version="1.0" encoding="UTF-8"?>
    <urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">
      <url>
        <loc>https://domainanda.com</loc>
        <lastmod>${new Date().toISOString()}</lastmod>
        <priority>1.0</priority>
      </url>
      ${cars.map(car => `
        <url>
          <loc>https://domainanda.com/cars/${car.id}</loc>
          <lastmod>${car.updatedAt}</lastmod>
          <priority>0.8</priority>
        </url>
      `).join('')}
    </urlset>
  `, {
    headers: {
      'Content-Type': 'application/xml',
    },
  })
}
```

---

## 🛠️ 8. Troubleshooting

### A. Common Issues
```bash
# 1. Database connection error
# Solution: Check DATABASE_URL dan permissions

# 2. Build error
# Solution: Clear .next folder dan rebuild
rm -rf .next
npm run build

# 3. Static file not found
# Solution: Check public folder dan paths

# 4. API 404 errors
# Solution: Check API routes dan middleware
```

### B. Performance Issues
```bash
# 1. Slow loading
# Solution: Enable compression dan caching

# 2. Memory issues
# Solution: Optimize images dan database queries

# 3. Database size
# Solution: Regular cleanup dan optimization
```

---

## 📋 9. Checklist Sebelum Go-Live

### Pre-Deployment Checklist:
- [ ] Build berhasil tanpa error
- [ ] Environment variables sudah di-set
- [ ] Database connection tested
- [ ] SSL certificate installed
- [ ] Domain DNS configured
- [ ] API endpoints tested
- [ ] Mobile responsiveness checked
- [ ] Performance optimized
- [ ] Security headers configured
- [ ] Backup strategy ready
- [ ] Monitoring setup
- [ ] Error handling tested

### Post-Deployment Checklist:
- [ ] Website accessible via domain
- [ ] All pages loading correctly
- [ ] Forms working properly
- [ ] Database operations working
- [ ] API responses correct
- [ ] No console errors
- [ ] Mobile functionality working
- [ ] Performance scores acceptable
- [ ] Security scan passed

---

## 🆘 10. Support & Maintenance

### A. Regular Maintenance
```bash
# 1. Update dependencies
npm update

# 2. Database backup
cp db/prod.db backups/$(date +%Y%m%d).db

# 3. Log rotation
pm2 logs rental-mobil --lines 1000 > logs/$(date +%Y%m%d).log

# 4. Performance monitoring
pm2 monit
```

### B. Emergency Procedures
```bash
# 1. Quick rollback
git checkout previous-stable-version
npm run build
pm2 restart rental-mobil

# 2. Database restore
cp backups/latest.db db/prod.db
pm2 restart rental-mobil

# 3. Error notification
# Setup alerts untuk downtime
```

---

## 📞 Kontak Support

Jika mengalami kendala selama deployment:
1. **Error Logs**: Check browser console dan server logs
2. **Database Issues**: Verify connection dan permissions
3. **Performance**: Monitor dengan Lighthouse
4. **Security**: Scan dengan security tools

---

## 🎉 Kesimpulan

Website rental mobil Anda siap untuk production! Pilih platform hosting yang sesuai dengan kebutuhan dan budget Anda:

- **Pemula**: Vercel/Netlify (Free & Mudah)
- **Menengah**: cPanel/Shared Hosting (Terjangkau)
- **Advanced**: VPS/Dedicated (Full control)

Ikuti checklist di atas untuk deployment yang sukses dan lancar!