'use client'

import { useState, useEffect } from 'react'
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { Label } from '@/components/ui/label'
import { Textarea } from '@/components/ui/textarea'
import { Badge } from '@/components/ui/badge'
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs'
import { 
  Car, 
  Users, 
  Calendar, 
  Settings, 
  Plus,
  Edit,
  Trash2,
  Save,
  X,
  TrendingUp,
  DollarSign,
  Activity
} from 'lucide-react'
import Link from 'next/link'

export default function AdminDashboard() {
  const [cars, setCars] = useState([])
  const [bookings, setBookings] = useState([])
  const [settings, setSettings] = useState(null)
  const [stats, setStats] = useState({
    totalCars: 0,
    totalBookings: 0,
    totalRevenue: 0,
    activeBookings: 0
  })
  const [isLoading, setIsLoading] = useState(true)
  const [editingCar, setEditingCar] = useState(null)
  const [editingSettings, setEditingSettings] = useState(false)

  useEffect(() => {
    fetchData()
  }, [])

  const fetchData = async () => {
    try {
      const [carsRes, bookingsRes, settingsRes] = await Promise.all([
        fetch('/api/cars'),
        fetch('/api/bookings'),
        fetch('/api/settings')
      ])

      const carsData = await carsRes.json()
      const bookingsData = await bookingsRes.json()
      const settingsData = await settingsRes.json()

      if (carsData.success) setCars(carsData.data)
      if (bookingsData.success) setBookings(bookingsData.data)
      if (settingsData.success) setSettings(settingsData.data)

      // Calculate stats
      setStats({
        totalCars: carsData.data?.length || 0,
        totalBookings: bookingsData.data?.length || 0,
        totalRevenue: bookingsData.data?.reduce((sum, booking) => sum + booking.totalPrice, 0) || 0,
        activeBookings: bookingsData.data?.filter(b => b.status === 'CONFIRMED').length || 0
      })
    } catch (error) {
      console.error('Failed to fetch data:', error)
    } finally {
      setIsLoading(false)
    }
  }

  const handleSaveCar = async (carData) => {
    try {
      const url = editingCar ? `/api/cars/${editingCar.id}` : '/api/cars'
      const method = editingCar ? 'PUT' : 'POST'
      
      const response = await fetch(url, {
        method,
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(carData)
      })

      if (response.ok) {
        setEditingCar(null)
        fetchData()
      }
    } catch (error) {
      console.error('Failed to save car:', error)
    }
  }

  const handleSaveSettings = async (settingsData) => {
    try {
      const response = await fetch('/api/settings', {
        method: 'PUT',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(settingsData)
      })

      if (response.ok) {
        setEditingSettings(false)
        fetchData()
      }
    } catch (error) {
      console.error('Failed to save settings:', error)
    }
  }

  const handleDeleteCar = async (carId) => {
    if (confirm('Apakah Anda yakin ingin menghapus mobil ini?')) {
      try {
        const response = await fetch(`/api/cars/${carId}`, {
          method: 'DELETE'
        })

        if (response.ok) {
          fetchData()
        }
      } catch (error) {
        console.error('Failed to delete car:', error)
      }
    }
  }

  if (isLoading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-600 mx-auto"></div>
          <p className="mt-4 text-gray-600">Loading...</p>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <header className="bg-white shadow-sm border-b">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <div className="flex items-center">
              <Link href="/" className="flex items-center space-x-2">
                <Car className="h-8 w-8 text-blue-600" />
                <span className="text-xl font-bold text-gray-900">Admin Panel</span>
              </Link>
            </div>
            <Link href="/">
              <Button variant="outline">View Website</Button>
            </Link>
          </div>
        </div>
      </header>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Stats Cards */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
          <Card>
            <CardContent className="p-6">
              <div className="flex items-center">
                <Car className="h-8 w-8 text-blue-600" />
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-600">Total Mobil</p>
                  <p className="text-2xl font-bold text-gray-900">{stats.totalCars}</p>
                </div>
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-6">
              <div className="flex items-center">
                <Calendar className="h-8 w-8 text-green-600" />
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-600">Total Booking</p>
                  <p className="text-2xl font-bold text-gray-900">{stats.totalBookings}</p>
                </div>
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-6">
              <div className="flex items-center">
                <DollarSign className="h-8 w-8 text-yellow-600" />
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-600">Total Revenue</p>
                  <p className="text-2xl font-bold text-gray-900">
                    Rp {stats.totalRevenue.toLocaleString('id-ID')}
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-6">
              <div className="flex items-center">
                <Activity className="h-8 w-8 text-purple-600" />
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-600">Booking Aktif</p>
                  <p className="text-2xl font-bold text-gray-900">{stats.activeBookings}</p>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Main Content */}
        <Tabs defaultValue="cars" className="space-y-6">
          <TabsList>
            <TabsTrigger value="cars">Manajemen Mobil</TabsTrigger>
            <TabsTrigger value="bookings">Booking</TabsTrigger>
            <TabsTrigger value="settings">Pengaturan</TabsTrigger>
          </TabsList>

          {/* Cars Management */}
          <TabsContent value="cars">
            <Card>
              <CardHeader>
                <div className="flex justify-between items-center">
                  <CardTitle>Manajemen Armada</CardTitle>
                  <Button onClick={() => setEditingCar({})}>
                    <Plus className="mr-2 h-4 w-4" />
                    Tambah Mobil
                  </Button>
                </div>
              </CardHeader>
              <CardContent>
                {editingCar && (
                  <div className="mb-6 p-4 border rounded-lg bg-gray-50">
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                      <div>
                        <Label htmlFor="name">Nama Mobil</Label>
                        <Input
                          id="name"
                          value={editingCar.name || ''}
                          onChange={(e) => setEditingCar({...editingCar, name: e.target.value})}
                        />
                      </div>
                      <div>
                        <Label htmlFor="brand">Merek</Label>
                        <Input
                          id="brand"
                          value={editingCar.brand || ''}
                          onChange={(e) => setEditingCar({...editingCar, brand: e.target.value})}
                        />
                      </div>
                      <div>
                        <Label htmlFor="model">Model</Label>
                        <Input
                          id="model"
                          value={editingCar.model || ''}
                          onChange={(e) => setEditingCar({...editingCar, model: e.target.value})}
                        />
                      </div>
                      <div>
                        <Label htmlFor="year">Tahun</Label>
                        <Input
                          id="year"
                          type="number"
                          value={editingCar.year || ''}
                          onChange={(e) => setEditingCar({...editingCar, year: e.target.value})}
                        />
                      </div>
                      <div>
                        <Label htmlFor="price">Harga per Hari</Label>
                        <Input
                          id="price"
                          type="number"
                          value={editingCar.price || ''}
                          onChange={(e) => setEditingCar({...editingCar, price: e.target.value})}
                        />
                      </div>
                      <div>
                        <Label htmlFor="seats">Jumlah Kursi</Label>
                        <Input
                          id="seats"
                          type="number"
                          value={editingCar.seats || ''}
                          onChange={(e) => setEditingCar({...editingCar, seats: e.target.value})}
                        />
                      </div>
                      <div>
                        <Label htmlFor="transmission">Transmisi</Label>
                        <select
                          id="transmission"
                          className="w-full p-2 border rounded"
                          value={editingCar.transmission || 'MANUAL'}
                          onChange={(e) => setEditingCar({...editingCar, transmission: e.target.value})}
                        >
                          <option value="MANUAL">Manual</option>
                          <option value="AUTOMATIC">Automatic</option>
                        </select>
                      </div>
                      <div>
                        <Label htmlFor="fuelType">Bahan Bakar</Label>
                        <select
                          id="fuelType"
                          className="w-full p-2 border rounded"
                          value={editingCar.fuelType || 'GASOLINE'}
                          onChange={(e) => setEditingCar({...editingCar, fuelType: e.target.value})}
                        >
                          <option value="GASOLINE">Bensin</option>
                          <option value="DIESEL">Diesel</option>
                          <option value="HYBRID">Hybrid</option>
                          <option value="ELECTRIC">Electric</option>
                        </select>
                      </div>
                      <div className="md:col-span-2">
                        <Label htmlFor="description">Deskripsi</Label>
                        <Textarea
                          id="description"
                          value={editingCar.description || ''}
                          onChange={(e) => setEditingCar({...editingCar, description: e.target.value})}
                        />
                      </div>
                    </div>
                    <div className="flex justify-end space-x-2 mt-4">
                      <Button variant="outline" onClick={() => setEditingCar(null)}>
                        <X className="mr-2 h-4 w-4" />
                        Batal
                      </Button>
                      <Button onClick={() => handleSaveCar(editingCar)}>
                        <Save className="mr-2 h-4 w-4" />
                        Simpan
                      </Button>
                    </div>
                  </div>
                )}

                <div className="overflow-x-auto">
                  <table className="w-full">
                    <thead>
                      <tr className="border-b">
                        <th className="text-left p-2">Nama</th>
                        <th className="text-left p-2">Merek</th>
                        <th className="text-left p-2">Model</th>
                        <th className="text-left p-2">Tahun</th>
                        <th className="text-left p-2">Harga</th>
                        <th className="text-left p-2">Status</th>
                        <th className="text-left p-2">Aksi</th>
                      </tr>
                    </thead>
                    <tbody>
                      {cars.map((car) => (
                        <tr key={car.id} className="border-b">
                          <td className="p-2">{car.name}</td>
                          <td className="p-2">{car.brand}</td>
                          <td className="p-2">{car.model}</td>
                          <td className="p-2">{car.year}</td>
                          <td className="p-2">Rp {car.price.toLocaleString('id-ID')}</td>
                          <td className="p-2">
                            <Badge variant={car.available ? 'default' : 'secondary'}>
                              {car.available ? 'Tersedia' : 'Tidak Tersedia'}
                            </Badge>
                          </td>
                          <td className="p-2">
                            <div className="flex space-x-2">
                              <Button
                                size="sm"
                                variant="outline"
                                onClick={() => setEditingCar(car)}
                              >
                                <Edit className="h-4 w-4" />
                              </Button>
                              <Button
                                size="sm"
                                variant="outline"
                                onClick={() => handleDeleteCar(car.id)}
                              >
                                <Trash2 className="h-4 w-4" />
                              </Button>
                            </div>
                          </td>
                        </tr>
                      ))}
                    </tbody>
                  </table>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Bookings */}
          <TabsContent value="bookings">
            <Card>
              <CardHeader>
                <CardTitle>Manajemen Booking</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="overflow-x-auto">
                  <table className="w-full">
                    <thead>
                      <tr className="border-b">
                        <th className="text-left p-2">ID</th>
                        <th className="text-left p-2">Pelanggan</th>
                        <th className="text-left p-2">Mobil</th>
                        <th className="text-left p-2">Tanggal</th>
                        <th className="text-left p-2">Total Harga</th>
                        <th className="text-left p-2">Status</th>
                      </tr>
                    </thead>
                    <tbody>
                      {bookings.map((booking) => (
                        <tr key={booking.id} className="border-b">
                          <td className="p-2">{booking.id.slice(0, 8)}</td>
                          <td className="p-2">{booking.user?.name || 'N/A'}</td>
                          <td className="p-2">{booking.car?.name}</td>
                          <td className="p-2">
                            {new Date(booking.startDate).toLocaleDateString('id-ID')} - {new Date(booking.endDate).toLocaleDateString('id-ID')}
                          </td>
                          <td className="p-2">Rp {booking.totalPrice.toLocaleString('id-ID')}</td>
                          <td className="p-2">
                            <Badge 
                              variant={
                                booking.status === 'CONFIRMED' ? 'default' :
                                booking.status === 'PENDING' ? 'secondary' :
                                booking.status === 'CANCELLED' ? 'destructive' : 'outline'
                              }
                            >
                              {booking.status}
                            </Badge>
                          </td>
                        </tr>
                      ))}
                    </tbody>
                  </table>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Settings */}
          <TabsContent value="settings">
            <Card>
              <CardHeader>
                <div className="flex justify-between items-center">
                  <CardTitle>Pengaturan Website</CardTitle>
                  <Button 
                    variant="outline" 
                    onClick={() => setEditingSettings(!editingSettings)}
                  >
                    {editingSettings ? 'Batal' : 'Edit'}
                  </Button>
                </div>
              </CardHeader>
              <CardContent>
                {editingSettings ? (
                  <div className="space-y-4">
                    <div>
                      <Label htmlFor="companyName">Nama Perusahaan</Label>
                      <Input
                        id="companyName"
                        value={settings?.companyName || ''}
                        onChange={(e) => setSettings({...settings, companyName: e.target.value})}
                      />
                    </div>
                    <div>
                      <Label htmlFor="phone">Telepon</Label>
                      <Input
                        id="phone"
                        value={settings?.phone || ''}
                        onChange={(e) => setSettings({...settings, phone: e.target.value})}
                      />
                    </div>
                    <div>
                      <Label htmlFor="email">Email</Label>
                      <Input
                        id="email"
                        type="email"
                        value={settings?.email || ''}
                        onChange={(e) => setSettings({...settings, email: e.target.value})}
                      />
                    </div>
                    <div>
                      <Label htmlFor="address">Alamat</Label>
                      <Input
                        id="address"
                        value={settings?.address || ''}
                        onChange={(e) => setSettings({...settings, address: e.target.value})}
                      />
                    </div>
                    <div>
                      <Label htmlFor="about">Tentang Kami</Label>
                      <Textarea
                        id="about"
                        rows={4}
                        value={settings?.about || ''}
                        onChange={(e) => setSettings({...settings, about: e.target.value})}
                      />
                    </div>
                    <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                      <div>
                        <Label htmlFor="facebook">Facebook</Label>
                        <Input
                          id="facebook"
                          value={settings?.facebook || ''}
                          onChange={(e) => setSettings({...settings, facebook: e.target.value})}
                        />
                      </div>
                      <div>
                        <Label htmlFor="instagram">Instagram</Label>
                        <Input
                          id="instagram"
                          value={settings?.instagram || ''}
                          onChange={(e) => setSettings({...settings, instagram: e.target.value})}
                        />
                      </div>
                      <div>
                        <Label htmlFor="twitter">Twitter</Label>
                        <Input
                          id="twitter"
                          value={settings?.twitter || ''}
                          onChange={(e) => setSettings({...settings, twitter: e.target.value})}
                        />
                      </div>
                    </div>
                    <div className="flex justify-end">
                      <Button onClick={() => handleSaveSettings(settings)}>
                        <Save className="mr-2 h-4 w-4" />
                        Simpan Pengaturan
                      </Button>
                    </div>
                  </div>
                ) : (
                  <div className="space-y-4">
                    <div>
                      <h3 className="font-semibold">Nama Perusahaan</h3>
                      <p className="text-gray-600">{settings?.companyName || 'N/A'}</p>
                    </div>
                    <div>
                      <h3 className="font-semibold">Kontak</h3>
                      <p className="text-gray-600">
                        {settings?.phone || 'N/A'} | {settings?.email || 'N/A'}
                      </p>
                    </div>
                    <div>
                      <h3 className="font-semibold">Alamat</h3>
                      <p className="text-gray-600">{settings?.address || 'N/A'}</p>
                    </div>
                    <div>
                      <h3 className="font-semibold">Tentang Kami</h3>
                      <p className="text-gray-600">{settings?.about || 'N/A'}</p>
                    </div>
                    <div>
                      <h3 className="font-semibold">Sosial Media</h3>
                      <div className="flex space-x-4 text-blue-600">
                        {settings?.facebook && <a href={settings.facebook}>Facebook</a>}
                        {settings?.instagram && <a href={settings.instagram}>Instagram</a>}
                        {settings?.twitter && <a href={settings.twitter}>Twitter</a>}
                      </div>
                    </div>
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </div>
  )
}