import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const featured = searchParams.get('featured')
    const limit = searchParams.get('limit')
    const page = searchParams.get('page') || '1'
    const search = searchParams.get('search')
    const brand = searchParams.get('brand')
    const transmission = searchParams.get('transmission')
    const fuelType = searchParams.get('fuelType')
    const minPrice = searchParams.get('minPrice')
    const maxPrice = searchParams.get('maxPrice')

    const where: any = {}
    
    if (featured === 'true') {
      where.available = true
    }
    
    if (search) {
      where.OR = [
        { name: { contains: search, mode: 'insensitive' } },
        { brand: { contains: search, mode: 'insensitive' } },
        { model: { contains: search, mode: 'insensitive' } }
      ]
    }
    
    if (brand) {
      where.brand = { contains: brand, mode: 'insensitive' }
    }
    
    if (transmission) {
      where.transmission = transmission
    }
    
    if (fuelType) {
      where.fuelType = fuelType
    }
    
    if (minPrice || maxPrice) {
      where.price = {}
      if (minPrice) where.price.gte = parseFloat(minPrice)
      if (maxPrice) where.price.lte = parseFloat(maxPrice)
    }

    const take = limit ? parseInt(limit) : undefined
    const skip = page && !limit ? (parseInt(page) - 1) * 10 : undefined

    const [cars, total] = await Promise.all([
      db.car.findMany({
        where,
        take,
        skip,
        orderBy: { createdAt: 'desc' }
      }),
      db.car.count({ where })
    ])

    return NextResponse.json({
      success: true,
      data: cars,
      pagination: limit ? undefined : {
        page: parseInt(page),
        total,
        pages: Math.ceil(total / 10)
      }
    })
  } catch (error) {
    console.error('Cars API error:', error)
    return NextResponse.json(
      { success: false, error: 'Internal server error' },
      { status: 500 }
    )
  }
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json()
    const {
      name,
      brand,
      model,
      year,
      price,
      image,
      description,
      transmission,
      fuelType,
      seats,
      available
    } = body

    if (!name || !brand || !model || !year || !price) {
      return NextResponse.json(
        { success: false, error: 'Required fields missing' },
        { status: 400 }
      )
    }

    const car = await db.car.create({
      data: {
        name,
        brand,
        model,
        year: parseInt(year),
        price: parseFloat(price),
        image,
        description,
        transmission: transmission || 'MANUAL',
        fuelType: fuelType || 'GASOLINE',
        seats: seats ? parseInt(seats) : 4,
        available: available !== undefined ? available : true
      }
    })

    return NextResponse.json({
      success: true,
      data: car
    })
  } catch (error) {
    console.error('Create car error:', error)
    return NextResponse.json(
      { success: false, error: 'Internal server error' },
      { status: 500 }
    )
  }
}