# 🚀 Panduan Deploy ke Vercel dengan Domain Sendiri

## 📋 Prasyarat

### 1. Akun Vercel
- Buat akun di [vercel.com](https://vercel.com)
- Install Vercel CLI: `npm i -g vercel`
- Login: `vercel login`

### 2. Domain Configuration
- Domain sudah dibeli (misal: `rentalmobilku.com`)
- Akses ke domain provider untuk konfigurasi DNS

---

## 🔧 Tahap 1: Setup Project

### A. Initialize Vercel Project
```bash
# Login ke Vercel
vercel login

# Link project ke Git (opsional)
git init
git add .
git commit -m "Initial commit"
git remote add origin https://github.com/username/rental-mobil.git
git push -u origin main

# Deploy ke Vercel
vercel --prod
```

### B. Update Konfigurasi Vercel
Setelah deploy pertama, Vercel akan otomatis membuat project. Update konfigurasi:

```bash
# Edit vercel.json
nano vercel.json
```

---

## 🔧 Tahap 2: Konfigurasi Domain Custom

### A. Tambah Domain di Vercel Dashboard

1. Buka [Vercel Dashboard](https://vercel.com/dashboard)
2. Pilih project rental-mobil-website
3. Klik tab "Domains"
4. Klik "Add Domain"
5. Masukkan domain Anda: `yourdomain.com`
6. Pilih "Production Environment"

### B. Konfigurasi DNS di Domain Provider

#### Option 1: CNAME Record (Recommended)
```
Type: CNAME
Name: @ (atau www)
Value: cname.vercel-dns.com
TTL: 300
```

#### Option 2: A Record (Jika tidak bisa CNAME)
```
Type: A
Name: @
Value: 76.76.21.21 (IP Vercel)
TTL: 300
```

### C. Konfigurasi Subdomain (www)
Untuk www.yourdomain.com:
```
Type: CNAME
Name: www
Value: cname.vercel-dns.com
TTL: 300
```

---

## 🔧 Tahap 3: Environment Variables

### A. Setup di Vercel Dashboard
1. Buka project settings
2. Klik "Environment Variables"
3. Tambahkan variables berikut:

```env
DATABASE_URL=file:./db/prod.db
NEXTAUTH_SECRET=your-super-secret-key-here
NEXTAUTH_URL=https://yourdomain.com
CUSTOM_DOMAIN_URL=https://yourdomain.com
NODE_ENV=production
```

### B. Update Local Configuration
```bash
# Update .env.production
cp .env.production.example .env.production

# Edit dengan nilai production
nano .env.production
```

---

## 🔧 Tahap 4: Build & Deploy

### A. Production Build
```bash
# Clean build sebelumnya
rm -rf .next

# Build untuk production
npm run build

# Test build lokal
npm run start
```

### B. Deploy ke Vercel
```bash
# Deploy dengan custom domain
vercel --prod --env DATABASE_URL,NEXTAUTH_SECRET,NEXTAUTH_URL,CUSTOM_DOMAIN_URL
```

---

## 🔧 Tahap 5: SSL & Security

### A. SSL Certificate (Otomatis)
Vercel otomatis menyediakan SSL certificate untuk:
- ✅ Custom domain
- ✅ Wildcard SSL
- ✅ Auto-renewal

### B. Security Headers
Sudah dikonfigurasi di `vercel.json`:
- X-Frame-Options: DENY
- X-Content-Type-Options: nosniff
- X-XSS-Protection: 1; mode=block
- Referrer-Policy: strict-origin-when-cross-origin

---

## 🔧 Tahap 6: Database Production

### A. Upload Database
```bash
# Upload database production ke Vercel
# Option 1: Gunakan Vercel Postgres (Recommended)
# Option 2: Upload SQLite file ke Vercel Storage
```

### B. Update Database Configuration
```javascript
// lib/db.ts untuk production
import { PrismaClient } from '@prisma/client'

const globalForPrisma = globalThis as unknown as {
  prisma: PrismaClient | undefined
}

export const db = globalForPrisma.prisma || new PrismaClient({
  datasources: {
    db: {
      url: process.env.DATABASE_URL,
    },
  },
})

if (process.env.NODE_ENV !== 'production') globalForPrisma.prisma = db
```

---

## 🔧 Tahap 7: Testing & Validation

### A. Production Testing
```bash
# Test semua endpoints
curl -X GET https://yourdomain.com/api/cars
curl -X POST https://yourdomain.com/api/bookings

# Test dengan browser
# Buka https://yourdomain.com
# Test mobile responsiveness
```

### B. Performance Check
```bash
# Install Lighthouse
npm install -g lighthouse

# Run performance audit
lighthouse https://yourdomain.com --view
```

### C. SEO Validation
```bash
# Test meta tags
curl -I https://yourdomain.com

# Check robots.txt
curl https://yourdomain.com/robots.txt

# Test sitemap
curl https://yourdomain.com/sitemap.xml
```

---

## 🔧 Tahap 8: Monitoring & Maintenance

### A. Vercel Analytics
1. Buka Vercel Dashboard
2. Klik "Analytics"
3. Monitor:
   - Page views
   - Unique visitors
   - Performance metrics
   - Error rates

### B. Error Monitoring
```javascript
// lib/monitoring.ts
export function logProductionError(error: Error, context: any) {
  console.error('Production Error:', {
    error: error.message,
    stack: error.stack,
    context,
    timestamp: new Date().toISOString(),
    url: window.location.href
  })
  
  // Send ke monitoring service
  if (typeof window !== 'undefined') {
    // Send ke Vercel analytics atau external service
    fetch('/api/log-error', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ error, context })
    })
  }
}
```

---

## 🔧 Tahap 9: Custom Domain Optimization

### A. Next.js Configuration
```javascript
// next.config.js
/** @type {import('next').NextConfig} */
const nextConfig = {
  output: 'standalone',
  experimental: {
    optimizeCss: true,
    optimizePackageImports: ['lucide-react', '@radix-ui/react-icons']
  },
  images: {
    domains: ['yourdomain.com', 'www.yourdomain.com'],
    formats: ['image/webp', 'image/avif']
  },
  compress: true,
  assetPrefix: process.env.NODE_ENV === 'production' ? undefined : '',
}

module.exports = nextConfig
```

### B. Meta Tags SEO
```javascript
// app/layout.tsx
export const metadata = {
  title: 'Rental Mobil Professional - Sewa Mobil Terpercaya',
  description: 'Layanan rental mobil terbaik dengan armada lengkap, harga terjangkau, dan pelayanan prima untuk kebutuhan perjalanan Anda.',
  keywords: 'rental mobil, sewa mobil, car rental, mobil murah, sewa mobil jakarta',
  authors: [{ name: 'Rental Mobil Team' }],
  creator: 'Rental Mobil',
  publisher: 'Rental Mobil',
  
  openGraph: {
    title: 'Rental Mobil Professional',
    description: 'Layanan rental mobil terbaik dengan armada lengkap',
    url: 'https://yourdomain.com',
    siteName: 'Rental Mobil',
    images: [
      {
        url: 'https://yourdomain.com/logo.svg',
        width: 1200,
        height: 630,
        alt: 'Rental Mobil Logo',
      },
    ],
    locale: 'id_ID',
    type: 'website',
  },
  
  twitter: {
    card: 'summary_large_image',
    title: 'Rental Mobil Professional',
    description: 'Layanan rental mobil terbaik',
    images: ['https://yourdomain.com/logo.svg'],
  },
  
  robots: {
    index: true,
    follow: true,
    googleBot: {
      index: true,
      follow: true,
      'max-video-preview': -1,
      'max-image-preview': 'large',
    },
  },
}
```

---

## 🔧 Tahap 10: Troubleshooting

### A. Common Issues

#### 1. Domain Not Pointing
```bash
# Check DNS propagation
nslookup yourdomain.com
dig yourdomain.com CNAME

# Check Vercel assignment
vercel domains ls
```

#### 2. Build Errors
```bash
# Clean build
rm -rf .next node_modules
npm install
npm run build

# Check logs
vercel logs
```

#### 3. Environment Variables Missing
```bash
# Check current variables
vercel env ls

# Add missing variables
vercel env add DATABASE_URL
vercel env add NEXTAUTH_SECRET
```

#### 4. Database Connection Issues
```bash
# Test database connection
echo $DATABASE_URL

# Check Prisma schema
npx prisma db pull
npx prisma generate
```

### B. Performance Issues

#### 1. Large Bundle Size
```bash
# Analyze bundle
npx @next/bundle-analyzer

# Optimize images
npx next-optimized-images
```

#### 2. Slow API Response
```bash
# Check function logs
vercel logs

# Monitor API performance
curl -w "@{time_total}" -o /dev/null -s https://yourdomain.com/api/cars
```

---

## 🔧 Tahap 11: Maintenance

### A. Regular Updates
```bash
# Update dependencies
npm update

# Security updates
npm audit fix

# Deploy updates
npm run build
vercel --prod
```

### B. Backup Strategy
```bash
# Database backup
npx prisma db pull
cp db/prod.db backups/$(date +%Y%m%d).db

# Code backup
git tag -a v$(date +%Y%m%d) -m "Production backup $(date)"
git push origin v$(date +%Y%m%d)
```

---

## 🎯 Checklist Sebelum Go-Live

### Pre-Deployment:
- [ ] Domain DNS pointing ke Vercel
- [ ] SSL certificate aktif
- [ ] Environment variables ter-set
- [ ] Build success tanpa error
- [ ] Database connection tested
- [ ] API endpoints working
- [ ] Mobile responsive
- [ ] Performance optimized

### Post-Deployment:
- [ ] Website accessible via domain
- [ ] HTTPS working properly
- [ ] All pages loading correct
- [ ] Forms submitting properly
- [ ] No console errors
- [ ] SEO meta tags correct
- [ ] Lighthouse score > 90

---

## 📞 Support

### Vercel Documentation:
- [Dashboard](https://vercel.com/dashboard)
- [CLI Commands](https://vercel.com/docs/cli)
- [Environment Variables](https://vercel.com/docs/concepts/projects/environment-variables)
- [Custom Domains](https://vercel.com/docs/concepts/projects/custom-domains)

### Jika ada masalah:
1. Check Vercel logs: `vercel logs`
2. Check deployment status: `vercel ls`
3. Restart deployment: `vercel --prod --force`
4. Contact support melalui Vercel dashboard

---

## 🎉 Deployment Commands Summary

```bash
# Quick deploy (setelah setup awal)
npm run build
vercel --prod

# Deploy dengan environment variables spesifik
vercel --prod --env DATABASE_URL,NEXTAUTH_SECRET,NEXTAUTH_URL

# Force redeploy (jika perlu)
vercel --prod --force
```

Website rental mobil Anda siap untuk deployment ke Vercel dengan domain sendiri! 🚀